// $Id: CMidiFileRead.cpp,v 1.2 2007/02/08 21:08:09 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CMidiFile.hpp"
using Exponent::Midi::CMidiFile;

//	===========================================================================
const char CMidiFile::CMIDI_FILE_EVENT_SIZES[] = 
{
	2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,		// 0x80 - 0x8F
	2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,		// 0x90 - 0x9F
	2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,		// 0xA0 - 0xAF
	2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,		// 0xB0 - 0xBF
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,		// 0xC0 - 0xCF
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,		// 0xD0 - 0xDF
	2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,		// 0xE0 - 0xEF
	0, 0, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,		// 0xF0 - 0xFF
};

//	===========================================================================
bool CMidiFile::readFile(CMidiSequence &sequence)
{
	// First read the head chunk
	SMTHDChunk headChunk;
	if (!readFileHeader(m_stream, headChunk))
	{
		return false;
	}

	// Now we read the tracks
	if (!readTrackChunks(m_stream, headChunk, sequence))
	{
		return false;
	}
	
	return true;
}

//	===========================================================================
bool CMidiFile::readFileHeader(CFileStream &stream, SMTHDChunk &chunk)
{
	// Check open
	if (!stream.isStreamOpen())
	{
		return false;
	}

	// Read in the header
	stream >> chunk.m_mthdChunk[0] >> chunk.m_mthdChunk[1] >> chunk.m_mthdChunk[2] >> chunk.m_mthdChunk[3] >> chunk.m_mthdSize;

	// Check its a valid MTHD chunk
	if (!isValidMTHDChunk(chunk))
	{
		return false;
	}
	
	// Read the extra header infromation
	stream >> chunk.m_fileType >> chunk.m_numberOfTracks >> chunk.m_timeFormat;

	// Sanity check the file type
	switch(chunk.m_fileType)
	{
		case 0:
			// Sanity check the file
			if (chunk.m_numberOfTracks != 1)
			{
				return false;
			}
		break;
		case 1:
			// Tis fine, we support this
		break;
		case 2:
			/* We dont support format 2 files */
			return false;
		break;
		default:
			/* We dont support unknown file formats */
			return false;
		break;
	}

	// We are done :)
	return true;
}

//	===========================================================================
bool CMidiFile::readTrackChunks(CFileStream &stream, SMTHDChunk &headChunk, CMidiSequence &sequence)
{
	// Some temporary variables for use during the loop
	long tracksRead = 0;
	SMTRKChunk chunk;

	// Loop until we have read everything, or until an error occurs
	while(!stream.hasErrorOccurred() && !stream.hasReachEndOfFile() && (tracksRead < headChunk.m_numberOfTracks))
	{
		// REad in the header parts
		stream >> chunk.m_mtrkChunk[0] >> chunk.m_mtrkChunk[1] >> chunk.m_mtrkChunk[2] >> chunk.m_mtrkChunk[3] >> chunk.m_mtrkSize;

		// Check if that read to the end
		if (stream.hasReachEndOfFile())
		{
			continue;
		}

		// Check its a valid MTHD chunk
		if (!isValidMTRKChunk(chunk))
		{
			// Then we skip the rest of the bytes
			stream.advanceStream(chunk.m_mtrkSize);
			continue;
		}

		// Then we are reading a new track
		tracksRead++;
		CMidiTrack *track = new CMidiTrack;
		
		// The midi event that we are going to read
		SMidiEvent midiEvent;
		unsigned char lastStatus = 0;

		// We increment the number of tracks read
		while((chunk.m_mtrkSize > 0) && !stream.hasErrorOccurred() && !stream.hasReachEndOfFile())
		{
			int sizeRead = 0;
			int dataRead = 0;
			
			// Get the delay time since the last event
			midiEvent.m_delaySinceLast = readVariableLengthUnsignedLong(stream, sizeRead);

			chunk.m_mtrkSize -= sizeRead;
			
			unsigned char statusByte;
			stream >> statusByte;
			chunk.m_mtrkSize--;

			// Now check for running status
			if (statusByte >= 0x80)
			{
				midiEvent.m_statusByte = statusByte;
				lastStatus			   = statusByte;
			}
			else
			{
				midiEvent.m_statusByte			 = lastStatus;
				midiEvent.m_midiData[dataRead++] = statusByte;
			}

			// REad the remaining data
			while(dataRead < CMIDI_FILE_EVENT_SIZES[midiEvent.m_statusByte - 0x80])
			{
				stream >> midiEvent.m_midiData[dataRead++];
				chunk.m_mtrkSize--;
			}

			// Now we want to setup these midi events
			if (midiEvent.m_statusByte == 0xF0 || midiEvent.m_statusByte == 0x7F)			// SYSTEM EXCLUSIVE EVENT
			{
				// System exclusive event
				int subSize                 = 0;
				unsigned long metaEventSize = readVariableLengthUnsignedLong(stream, subSize);

				// For the moment we ignore these events
				stream.advanceStream(metaEventSize);
				chunk.m_mtrkSize -= metaEventSize;
			}
			else if (midiEvent.m_statusByte == 0xFF)										// META EVENT							
			{
				// Read some data from the stream
				stream >> midiEvent.m_midiData[0];
				chunk.m_mtrkSize--;

				int subSize					= 0;
				unsigned long metaEventSize = readVariableLengthUnsignedLong(stream, subSize);
				chunk.m_mtrkSize		   -= subSize;

				if (midiEvent.m_midiData[0] == CMidi::CMIDI_MIDIFILE_TEMPO)
				{
					if (metaEventSize == 3)
					{
						// Set the tempo
						unsigned char tempoData[3];

						// Read the data
						stream >> tempoData[0] >> tempoData[1] >> tempoData[2];

						unsigned long tempo = 0;

						for (long i = 0; i < 3; i++)
						{
							tempo <<= 8;
							tempo += tempoData[i];
						}

						// Microseconds per quarter note
						sequence.setTempo(60000000.0 / (double)tempo);
					}
				}
				else if (midiEvent.m_midiData[0] == CMidi::CMIDI_MIDIFILE_TIME_SIGNATURE)
				{
					if (metaEventSize == 4)
					{
						// Time signature
						unsigned char timeSignature[4];

						// Stream in
						stream >> timeSignature[0] >> timeSignature[1] >> timeSignature[2] >> timeSignature[3];

						// Store it, noting that the denominator is 2^
						sequence.setTimeSignature(CTimeSignature((long)timeSignature[0], (1 << timeSignature[1])));
					}
				}
				else if (midiEvent.m_midiData[0] == CMidi::CMIDI_MIDIFILE_TRACK_NUM)
				{
					if (metaEventSize == 1)
					{
						unsigned char midiChannel = 0;
						stream >> midiChannel;

						// Set the midi channel
						track->setMidiChannel((long)midiChannel);
					}
				}
				else if (midiEvent.m_midiData[0] == CMidi::CMIDI_MIDIFILE_TRACK_NAME)
				{
					unsigned char *nameBuffer = new unsigned char[metaEventSize + 1];
					memset(nameBuffer, 0, metaEventSize * sizeof(unsigned char));
					stream.readDataFromStream(nameBuffer, metaEventSize * sizeof(unsigned char));
					nameBuffer[metaEventSize] = '\0';
					track->setTrackName(CString((const char *)nameBuffer));
					FREE_ARRAY_POINTER(nameBuffer);
				}
				else
				{
					stream.advanceStream(metaEventSize);
				}
				chunk.m_mtrkSize -= metaEventSize;
			}
			else
			{
				// Read the midi events in heere...
				CMidiEvent *event = NULL;

				// Its a normal note....
				// We need to handle the normal way
				unsigned char theStatus = midiEvent.m_statusByte & CMidi::CMIDI_STATUS_MASK;
				switch(theStatus)
				{
					case CMidi::CMIDI_NOTE_ON:
					case CMidi::CMIDI_NOTE_OFF:
						{
							const long note     = midiEvent.m_midiData[0] & CMidi::CMIDI_VALUE_MASK;
							const long velocity = midiEvent.m_midiData[1] & CMidi::CMIDI_VALUE_MASK;

							// We need to set the midi channel and the delta time here!!!!
							event = new CMidiEvent(theStatus, note, velocity, 0, midiEvent.m_delaySinceLast);
						}
					break;
					case CMidi::CMIDI_CONTROL_CHANGE:
						if (midiEvent.m_midiData[1] == CMidi::CMIDI_MONO_MODE_ON || midiEvent.m_midiData[1] == CMidi::CMIDI_ALL_NOTES_OFF)
						{
							event = new CMidiEvent(CMidi::CMIDI_ALL_NOTES_OFF, 0, 0, 0, midiEvent.m_delaySinceLast);
						}
						else
						{
							const long controller = midiEvent.m_midiData[0] & CMidi::CMIDI_VALUE_MASK;
							const long value      = midiEvent.m_midiData[1] & CMidi::CMIDI_VALUE_MASK;
							event = new CMidiEvent(theStatus, controller, value, 0, midiEvent.m_delaySinceLast);
						}
					break;
					case CMidi::CMIDI_PITCH_BEND:
						event = new CMidiEvent(CMidi::CMIDI_PITCH_BEND, 0, 0, 0, midiEvent.m_delaySinceLast);
						event->setPitchBend(((double)(((unsigned char)midiEvent.m_midiData[0]) + (((unsigned char)midiEvent.m_midiData[1]) << 7)) / (double)0x2000) - 1.0);
					break;
				}

				// Check its valid
				if (event)
				{
					// Add the midi event to the track
					track->addEvent(event);
				}
			}
		}

		if (track->getNumberOfEvents() > 0)
		{
			// Now add the track to the sequence
			sequence.addTrack(track);
		}
		else
		{
			FREE_POINTER(track);
		}
	}

	// Now we want to process the sequence
	processMidiSequence(headChunk, sequence);

	// We are done!
	return true;
}

//	===========================================================================
unsigned long CMidiFile::readVariableLengthUnsignedLong(CFileStream &stream, int &sizeRead)
{	
	// Reset the size
	sizeRead = 0;

	// Some temporary variables
	unsigned long endOfValue  = 0;
	unsigned long returnValue = 0;
	char character			  = 0;

	do
	{
		stream >> character;
		endOfValue = static_cast<unsigned long>(character) & 255;
		sizeRead++;

		// Check its not unterminated
		if (sizeRead > 6)
		{
			return 0;
		}	
		returnValue = (returnValue << 7) + (endOfValue & 0x7F);
	}
	while(endOfValue & 0x80);

	// We are done! :)
	return returnValue;
}

//	===========================================================================
void CMidiFile::computeSamplesPerTick(SMTHDChunk &headChunk, const double sampleRate, const double bpm)
{
	if (headChunk.m_timeFormat < 0)
	{
		// SMPTE
		double framesPerSecond = 0.0;
		switch((-headChunk.m_timeFormat) >> 8)
		{
			case 24:	framesPerSecond = 24.0;		break;
			case 25:	framesPerSecond = 25.0;		break;
			case 29:	framesPerSecond = 29.97;	break;
			case 30:	framesPerSecond = 30.0;		break;
			default:
				/* Error condition, assume 30fps */
				framesPerSecond = 30.0;		
			break;
		}
		headChunk.m_samplesPerTick = sampleRate / framesPerSecond / (double)(headChunk.m_timeFormat & 0xFF);
	}
	else
	{
		// Tempo based
		const static double bpsDivisor = 0.016666666666666666666666666666667;	// 1 / 60
		double beatsPerSecond = bpm * bpsDivisor;
		headChunk.m_samplesPerTick = sampleRate / beatsPerSecond / (double)headChunk.m_timeFormat;
	}
}

//	===========================================================================
void CMidiFile::processMidiSequence(SMTHDChunk &headChunk, CMidiSequence &sequence)
{
	// Compute the sample delay time
	computeSamplesPerTick(headChunk, sequence.getSampleRate(), sequence.getTempo());

	// At this point the tracks contain the events in 
	for (long i = 0; i < sequence.getNumberOfTracks(); i++)
	{
		// Get the track
		CMidiTrack *track = sequence.getMutableTrackAtIndex(i);

		// We can only handle
		if (track == NULL)
		{
			continue;
		}

		// Delta time is cummulative
		long lastEventPosInTicks = 0;

		// Now we need to process the events internally to update them to the correct delta time
		// Currently they are in the order of those on the disk file
		for (long j = 0; j < track->getNumberOfEvents(); j++)
		{
			// Get the event
			CMidiEvent *event = track->getMutableEventAtIndex(j);

			// Check its valid
			if (event == NULL)
			{
				continue;
			}

			// Ohterwise we need to find the relative value to the last one
			lastEventPosInTicks += event->getTimeDelta();
			event->setTimeDelta(lastEventPosInTicks);
		}

		double samplesRemainingFraction = 0.0;

		// The tracks events are now in absolute tick position
		// We now need to convert this absolute position to a delta time offset from 0 in samples rather than ticks
		for (long j = 0; j < track->getNumberOfEvents(); j++)
		{
			// Get the event
			CMidiEvent *event = track->getMutableEventAtIndex(j);

			// Check its valid
			if (event == NULL)
			{
				continue;
			}

			// Compute the number of samples in
			double samples = event->getTimeDelta() * headChunk.m_samplesPerTick + samplesRemainingFraction;
			event->setTimeDelta(static_cast<long>(samples));
			samplesRemainingFraction = samples - event->getTimeDelta();
		}

		// Now they are in sample order, we would hope!
	}

	sequence.compact();
}